#!/bin/bash

if [ -n "$debug" ] ; then
    set -x
fi

markdown="markdown"
SRCDIR="$1"
BASE_OUTPUT="$2"

pushd "$SRCDIR" > /dev/null


function header
{
    local title="$1"
    local breadcrumb="$2"
    local output="$3"

    cat - > $output <<EOF
<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML//EN">
<html>
<head>
        <title>$title</title>

        <style>
        * { margin: 0; padding: 0; }
        body { font-family: sans-serif; font-size: 12px; }
        #head { padding: 20px; background: #00A500; }
        #head a { color: #000; }
        #body { padding: 20px; }
        #foot { padding: 10px; font-size: 9px; border-top: 1px #18A303 solid; margin-top: 25px; }
        p { line-height: 1.7em; margin-bottom: 1em; }
        pre { margin-bottom: 0.5em; }
        .multi-col { -moz-column-width: 20em; -webkit-column-width: 20em; -moz-column-gap: 1em; -webkit-column-gap: 1em; }
        h1 { margin-bottom: 0.5em; }
        h2,h3,h4 { margin: 1.3em 0 0.5em 0; }
        ul, ol { margin: 0.5em 1.5em; }
        </style>
</head>
<body>
        <div id="head">
        <h1>${title}</h1>
        <p>${breadcrumb}</p>
        </div>
        <div id="body" style="multi-col">
EOF
}

function footer
{
local output="$1"

  cat - >> $output <<EOF

</div>
<div id="foot">
  <small>
    <p>Generated by Libreoffice CI on $(hostname)</p>
    <p>Last updated:
EOF

date '+%F %T' >> $output
cat - >> $output <<EOF
    | <a href="http://www.documentfoundation.org/privacy">Privacy Policy</a> | <a href="http://www.documentfoundation.org/imprint">Impressum (Legal Info)</a>
    </p>
  </small>
</div>
</body>
</html>
EOF

}

function proc_text
{
    # Local links: [[...]]
    # Git links: [git:...]
    # Other remote links: [...]
    # Headings: == bleh ==
    # Paragraphs: \n\n
    sed -re ' s/\[\[([-_a-zA-Z0-9]+)\]\]/<a href="\1.html">\1<\/a>/g' - \
        | sed -re ' s/\[git:([^]]+)\]/<a href="https:\/\/cgit.freedesktop.org\/libreoffice\/core\/tree\/\1">\1<\/a>/g' \
        | sed -re ' s/\[([^]]+)\]/<a href="\1">\1<\/a>/g' \
        | sed -re ' s/====([^=]+)====/<h4>\1<\/h4>/g' \
        | sed -re ' s/===([^=]+)===/<h3>\1<\/h3>/g' \
        | sed -re ' s/==([^=]+)==/<h2>\1<\/h2>/g' \
        | sed -re ':a;N;$!ba;s/\n\n/<\/p><p>/g' \
        | awk 'BEGIN { print "<p>" } { print } END { print "</p>" }'
}

function proc_text_markdown {
  sed -re ' s/\[git:([^]]+)\]/<a href="https:\/\/cgit.freedesktop.org\/libreoffice\/core\/tree\/\1">\1<\/a>/g'
}

# generate entry page

echo "generating index page"
header "LibreOffice Modules" " " "$BASE_OUTPUT/index.html"

for module_name in *; do
    if [ -d $module_name ]; then
        cur_file=$(echo $module_name/README.md)
        if [ -f "$cur_file" ]; then
            # write index.html entry
            text=$(echo -e "<h2><a href=\"${module_name}.html\">${module_name}</a></h2>\n")
            if [ ${cur_file: -3} == ".md" ]; then
                # This is a markdown file.
                header_text="$(head -n1 $cur_file)"
                header_text="$(echo ${header_text} | sed -e 's/^\#*//g')"
                text="${text}${header_text}"
            else
                text="${text}$(head -n1 $cur_file | proc_text)"
            fi
            echo -e "$text" >> "$BASE_OUTPUT/index.html"

            # write detailed module content
            header "$module_name" "<a href=\"index.html\">LibreOffice</a> &raquo; ${module_name}" "$BASE_OUTPUT/${module_name}.html"
            text="<p><b>View module in:</b>"
            text="${text} &nbsp; <a href=\"https://cgit.freedesktop.org/libreoffice/core/tree/${module_name}\">cgit</a>"
            if $(echo $INPUT_PROJECTS | grep -q $module_name); then
                text="${text} &nbsp; <a href=\"${module_name}/html/classes.html\">Doxygen</a>"
            fi
            text="${text} </p><p>&nbsp;</p>"
            echo -e "$text" >> "$BASE_OUTPUT/${module_name}.html"

        if [ ${cur_file: -3} == ".md" ]; then
            # This is a markdown file.
            text="$(${markdown} $cur_file | proc_text_markdown)"
            echo -e "$text" >> "$BASE_OUTPUT/${module_name}.html"
        else
            proc_text < $cur_file >> "$BASE_OUTPUT/${module_name}.html"
        fi
            footer "$BASE_OUTPUT/${module_name}.html"
        else
            empty_modules[${#empty_modules[*]}]=$module_name
        fi
    fi
done

if [ ${#empty_modules[*]} -gt 10 ]; then
    echo -e "<p>&nbsp;</p><p>READMEs were not available for these modules:</p><ul>\n" >> "$BASE_OUTPUT/index.html"
    for module_name in "${empty_modules[@]}"; do
        if [[ "$module_name" =~ ^(autom4te.cache|dictionaries|docs|helpcompiler|helpcontent2|include|instdir|lo|translations|workdir)$ ]]
        then
            continue
        fi
        echo -e "<li><a href=\"https://cgit.freedesktop.org/libreoffice/core/tree/${module_name}\">${module_name}</a></li>\n" >> "$BASE_OUTPUT/index.html"
    done
    echo -e "</ul>\n" >> "$BASE_OUTPUT/index.html"
fi

footer "$BASE_OUTPUT/index.html"

popd > /dev/null

## done
