/*******************************************************************************
* input.cpp: Input streams
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: input.cpp,v 1.2 2001/11/21 18:51:02 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include "config.h"

#include "../core/core.h"

#include "../mpeg/mpeg.h"
#include "../mpeg/ts.h"

#include "program.h"
#include "buffer.h"
#include "broadcast.h"
#include "output.h"
#include "channel.h"
#include "request.h"

#include "input.h"


/*******************************************************************************
* E_Input
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
E_Input::E_Input(const C_String& strInputName, const C_String& strMsg) :
            E_Exception(GEN_ERR, strInputName + ": " + strMsg)
{
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
E_Input::E_Input(const C_String& strInputName, const C_String& strMsg,
                 const E_Exception& e) :
            E_Exception(GEN_ERR, strInputName + ": " + strMsg, e)
{
}


/*******************************************************************************
* C_Input
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Input::C_Input(C_Module* pModule,
                 const C_String& strName) : m_strName(strName)
{
  ASSERT(pModule);

  m_pModule = pModule;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Input::SetEventHandler(C_EventHandler* pEventHandler)
{
  ASSERT(pEventHandler);

  m_pEventHandler = pEventHandler;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Input::C_Input(const C_Input& cInput)
{
  // Input must not be copied
  ASSERT(false);
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Input::~C_Input()
{
  m_pModule->Unref();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_Input::Init()
{
  LogDbg(NULL, "Initialising input "+m_strName);

  try
  {
    C_Application* pApp = C_Application::GetApp();
    ASSERT(pApp);

    // First register the source as a log client
    m_hLog = pApp->StartLog(m_strName, LOG_DBGMSG|LOG_FILE|LOG_SCR);
    ASSERT(m_hLog);
  
    // Now call the child's initialisation method
    OnInit();
  
    // Undo what has been done during initialisation to release the
    // ressource that may have been allocated at that time
  }
  catch(E_Exception e)
  {
    Destroy();
    
    throw E_Input(m_strName, "Unable to initialise input: aborting", e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Input::Destroy()
{
  LogDbg(NULL, "Destroying input "+m_strName);

  try
  {
    // Do some child specific cleanings
    OnDestroy();

    // Unregister the input by the logger
    C_Application* pApp = C_Application::GetApp();
    ASSERT(pApp);
    pApp->StopLog(m_hLog);
  }
  catch(E_Exception e)
  {
    throw E_Input(m_strName, "Unable to destroy input", e);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Input::StartStreaming(C_Broadcast* pBroadcast)
{
  C_Answer cAnswer(GetName());

  const C_Program* pPgrm = pBroadcast->GetProgram();
  const C_String strPgrmName = pPgrm->GetName();

  try
  {
    // Start the reception of the program
    LogDbg(m_hLog, "Starting reception of pgrm "+strPgrmName);
    OnStartStreaming(pBroadcast);

    // Update broadcast status
    pBroadcast->SetStatus(BROADCAST_RUNNING);

    // Build the answer
    cAnswer.SetStatus(NO_ERR);
    cAnswer.AddMessage("Program " + strPgrmName + " started");
  }
  catch(E_Exception e)
  {
    // Update broadcast status
    pBroadcast->SetStatus(BROADCAST_ERROR);

    // Build the answer
    cAnswer.SetStatus(e.GetCode());
    cAnswer.AddMessage("Unable to start program "+strPgrmName);
    cAnswer.AddMessage(e.Dump());
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Input::ResumeStreaming(C_Broadcast* pBroadcast)
{
  C_Answer cAnswer(GetName());

  const C_Program* pPgrm = pBroadcast->GetProgram();
  const C_String strPgrmName = pPgrm->GetName();

  try
  {
    // Ask the child to resume the streaming
    if(pBroadcast->GetStatus() == BROADCAST_SUSPENDED)
    {
      LogDbg(m_hLog, "Resuming reception of pgrm "+strPgrmName);
      OnResumeStreaming(pBroadcast);
      pBroadcast->SetStatus(BROADCAST_RUNNING);
      LogDbg(m_hLog, "Pgrm "+strPgrmName+" resumed");
    }
    else
    {
      cAnswer.AddMessage("Program " + strPgrmName + " isn't suspended");
    }
  }
  catch(E_Exception e)
  {
    // Update broadcast status
    pBroadcast->SetStatus(BROADCAST_ERROR);

    // Build the answer
    cAnswer.SetStatus(e.GetCode());
    cAnswer.AddMessage("Unable to resume program "+strPgrmName);
    cAnswer.AddMessage(e.Dump());
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Input::SuspendStreaming(C_Broadcast* pBroadcast)
{
  C_Answer cAnswer(GetName());

  const C_Program* pPgrm = pBroadcast->GetProgram();
  const C_String strPgrmName = pPgrm->GetName();

  try
  {
    // Ask the child to suspend the streaming
    if(pBroadcast->GetStatus() == BROADCAST_RUNNING)
    {
      LogDbg(m_hLog, "Suspending reception of pgrm "+strPgrmName);
      OnSuspendStreaming(pBroadcast);
      pBroadcast->SetStatus(BROADCAST_SUSPENDED);
      LogDbg(m_hLog, "Pgrm "+strPgrmName+" suspended");
    }
    else
    {
      cAnswer.AddMessage("Program " + strPgrmName + " wasn't running");
    }
  }
  catch(E_Exception e)
  {
    // Update broadcast status
    pBroadcast->SetStatus(BROADCAST_ERROR);

    // Build the answer
    cAnswer.SetStatus(e.GetCode());
    cAnswer.AddMessage("Unable to suspend program "+strPgrmName);
    cAnswer.AddMessage(e.Dump());
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Input::StopStreaming(C_Broadcast* pBroadcast)
{
  C_Answer cAnswer(GetName());
  cAnswer.SetStatus(NO_ERR);

  const C_Program* pPgrm = pBroadcast->GetProgram();
  const C_String strPgrmName = pPgrm->GetName();

  try
  {
    // Ask the child to stop the streaming
    if(pBroadcast->GetStatus() == BROADCAST_RUNNING)
    {
      LogDbg(m_hLog, "Stopping reception of pgrm "+strPgrmName);
      OnStopStreaming(pBroadcast);
      pBroadcast->SetStatus(BROADCAST_STOPPED);
      LogDbg(m_hLog, "Pgrm "+strPgrmName+" stopped");
    }
    else
    {
      cAnswer.AddMessage("Program " + strPgrmName + " not broadcasted");
    }
  }
  catch(E_Exception e)
  {
    LogDbg(m_hLog, "Error when stopping program "+strPgrmName);
    cAnswer.AddMessage("Error when stopping program " + strPgrmName);
    cAnswer.SetStatus(e.GetCode());
  }

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_List<C_Program> C_Input::GetAvailablePgrms()
{
  return OnGetAvailablePgrms();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*C_Answer C_Input::BrowsePgrmTable()
{
  C_Answer cAnswer(m_strName);
  cAnswer.SetStatus(NO_ERR);
  cAnswer.AddMessage("Program Table");

  m_cDirectory.Lock();
  for(unsigned int i = 0; i < m_cDirectory.GetSize(); i++)
  {
    C_Program& cPgrm = (m_cDirectory)[i];

    C_String strPgrmName = cPgrm.GetName();
    ASSERT(strPgrmName != "");

    C_Answer* pPgrmDescr = new C_Answer(strPgrmName);
    ASSERT(pPgrmDescr);
    pPgrmDescr->SetStatus(cPgrm.GetStatus());
    pPgrmDescr->AddMessage(cPgrm.GetDescription());

    cAnswer.Add(pPgrmDescr);
  }
  m_cDirectory.UnLock();

  return cAnswer;
}*/



//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*int C_Input::CancelStreaming(const C_Broadcast& cBroadcast)
{
  LogDbg(m_hLog, "Received internal stop command for pgrm "+iPgrmNumber);

  // Just lock the pgrm before asking the child to do the job
  C_Program* pPgrm = m_cDirectory.GetPgrm(iPgrmNumber);
  ASSERT(pPgrm);
  int iRc = StopReception(pPgrm);

  m_cDirectory.ReleasePgrm(pPgrm);
  
  return iRc;
}*/


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*int C_Input::UpdatePgrmTable()
{
  LogDbg(m_hLog, "Updating pgrm table");

  // Just lock the pgrm table before asking the child to do the job
  m_cDirectory.Lock();
  int iRc = OnUpdatePgrmTable();
  m_cDirectory.UnLock();

  return iRc;
}*/


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*int C_Input::Fail(int iError, const C_String& strMsg)
{
  LogDbg(m_hLog, "Input failure: " + strMsg);

  // Lock the pgrm table before recording the error and asking the child to
  // handle the pb
  m_cDirectory.Lock();
  m_iStatus = iError;
  m_strErrorMsg = strMsg;
  int iRc = OnFailure(iError);
  m_cDirectory.UnLock();

  return iRc;
}*/


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*int C_Input::Recover()
{
  LogDbg(m_hLog, "Input recovery");

  // Lock the pgrm table before recording the recovery and asking the child to
  // handle it
  m_cDirectory.Lock();
  m_iStatus = NO_ERR;
  m_strErrorMsg = "";
  int iRc = OnRecovery();
  m_cDirectory.UnLock();

  return iRc;
}*/


//------------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------
/*int C_Input::StartReception(C_Program* pPgrm, const C_Broadcast& cBroadcast)
{
  ASSERT(pPgrm);
  
  u16 iPgrmNumber = pPgrm.GetPgrmNumber();
  LogDbg(m_hLog, "Starting reception of pgrm "+iPgrmNumber);

  int iRc = OnStartReception(cBroadcast);
  
  if(!iRc)
  {
    LogDbg(m_hLog, "Reception of pgrm "+iPgrmNumber+" started");
    pPgrm->SetStatus(PGRM_BROADCASTED);
  }
  else
  {
    LogDbg(m_hLog, "Reception of pgrm "+iPgrmNumber+" failed");
    pPgrm->SetStatus(PGRM_ERROR);
    pOutput->Close();
  }
    
  return iRc;
}*/



//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
/*int C_Input::StopReception(C_Program* pPgrm, const C_Broadcast& cBroadcast)
{
  ASSERT(pPgrm);

  int iRc = NO_ERR;

  if(pPgrm->GetStatus() != PGRM_BROADCASTED)
  {
    // Program has already been stopped, there is nothing to do
    u16 iPgrmNumber = pPgrm->GetPgrmNumber(); 
    LogDbg(m_hLog, "Pgrm "+iPgrmNumber+" not broadcasted, no need to stop it");
  }
  else
  {
    u16 iPgrmNumber = pPgrm->GetPgrmNumber();

    LogDbg(m_hLog, "Stopping pgrm "+iPgrmNumber);
    iRc = OnStopReception(iPgrmNumber);

    if(!iRc)
      pPgrm->SetStatus(PGRM_AVAILABLE);
    else
      pPgrm->SetStatus(PGRM_ERROR);

    LogDbg(m_hLog, "Closing output for pgrm "+iPgrmNumber);
  }
  
  return iRc;
}*/


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
const C_String& C_Input::GetName() const
{
  return m_strName;
}


bool C_Input::operator == (const C_Input& cInput) const
{
  // The test is made on the input address: since all remote input have a local
  // instance of the C_RemoteInput class, this will work in all cases
  return (this == &cInput);
}


